### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ dfdaa2c2-58bc-4911-8e3b-de854e5d56fa
import Pkg; Pkg.activate(".")

# ╔═╡ f419b967-f090-4f5a-9db7-3f9f1a80f8bb
begin
	using CairoMakie
	using CommonMark
	using PlutoUI
end

# ╔═╡ d0fdd5a1-5989-471c-86f4-4516645331e6
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ 8eac6af6-0e5c-43c7-8781-e97b02d3813f
md"""
**This is a `julia` notebook**
"""

# ╔═╡ b67d9fd5-435e-4150-902e-e6915efdbeaa
Pkg.instantiate()

# ╔═╡ d34ab604-27fa-4877-a8e1-d17bb8fb026f
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ 48232ae4-81b0-4cf4-80a9-1a0085ba8d9f
md"""
# A few important Fourier Transform
***

## The rectangular function
***

- We want to compute the FT of the rectangular function: e.g. a function $g(t)$ with a given amplitude and extending from $t=-T/2$ to $t=T/2$. For $|t|>T/2$, $g(t)=0$.
  
- We will write the square pulse or box function as $rect_T(t)$, indicating that the rectangle function is equal to 1 for a period of $T$ (from $-T/2$ to $+T/2$) and 0 elsewhere,
"""

# ╔═╡ e5b7c371-b30d-44d3-8ee9-f347719ba366
md"""
 ```math
\mathfrak{F} \{  g(t) \} = G(f) = \int_{-\infty}^\infty g(t) e^{-2 \pi i f t} dt = \int_{-T/2}^{T/2} A e^{-2 \pi i f t} dt = \frac{A}{-2 \pi i f} e^{-2 \pi i f t}\Big|_{-T/2}^{T/2} = \frac{A}{-2 \pi i f} [ e^{-\pi i f T} - e^{\pi i f T} ] = 
```

```math
= \frac{AT}{\pi f T} \frac{ e^{-\pi i f T} - e^{\pi i f T} }{2i} = \frac{AT}{\pi f T} \sin (\pi f T) \equiv AT {\rm sinc} (fT)
```

 - where we have defined the "sinc" function defined as ${\rm sinc} (t) = \sin(\pi t) / \pi t$.
"""

# ╔═╡ fc298404-78c0-49b9-8250-091dbc2b7487
begin
	T=2.
	
	#f(x) = (x >= -T/2 && x <= T/2) ? 1 : 0
	
	function f(x; T=2)
	    if x >= -T/2 && x <= T/2
	        return 1
	    else
	        return 0
	    end
	end
	
	xrng = range(start=-5.,stop=5.,step=0.01)
	
	
	fg1 = Figure(size=(1000,500))
	
	ax1 = Axis(fg1[1, 1],
	    xlabel = "t",
	    )
	
	T=1
	lines!(xrng,map(x -> f(x;T=T),xrng),label="T = "*string(T))
	T=3
	lines!(xrng,map(x -> f(x;T=3),xrng),label="T = "*string(T))
	
	
	axislegend()
	
	
	frng = range(start=-10.,stop=10.,step=0.01)
	
	
	ax2 = Axis(fg1[1, 2],
	    xlabel = "f",
	    )
	
	T=1
	lines!(frng,T*sinc.(T*frng),label="T = "*string(T))
	T=3
	lines!(frng,T*sinc.(T*frng),label="T = "*string(T))
	
	xlims!(-5,5)
	
	axislegend()
	
	fg1
end

# ╔═╡ 9b5f3662-37da-4e19-a4a3-a80dbbfb041a
cm"""
## The Gaussian function
***

- We want now to compute the FT of the Gaussian function: ``g(t) = e^{-\pi t^2}``.

- Let G(f) be the Fourier Transform of g(t), so that:


```math
G(f) = \mathfrak{F} \{  g(t) \} = \int_{-\infty}^\infty g(t) e^{-2 \pi i f t} dt = \int_{-\infty}^\infty e^{-\pi t^2} e^{-2 \pi i f t} dt
```

- Take the derivative of both sides of equation with respect to f, so that:

```math
\frac{dG(f)}{dt} = \int_{-\infty}^\infty e^{- \pi t^2} (-2 i \pi t) e^{-2 \pi i f t} dt
```

- Writing ``u = e^{-2 \pi i f t}  \rightarrow du = (-2 \pi i f)e^{-2 \pi i f t} dt`` and ``v = i e^{-\pi t^2} \rightarrow dv = -2 \pi i t e^{-\pi t^2} dt ``, and given the formula for integration by parts: ``\int udv = uv - \int vdu`` we have (the uv term becomes zero, because the limits are evaluated from ``-\infty`` to ``\infty``, where the product is zero):

```math
\frac{dG(f)}{dt} = \int_{-\infty}^\infty ie^{- \pi t^2} (-2 i \pi f) e^{-2 \pi i f t} dt = -2 \pi f \int_{-\infty}^\infty e^{- \pi t^2} e^{-2 \pi i f t} dt = -2 \pi f G(t)
```

- This is a first order simple differential equation for ``G(f)``. The solution for this differential equation is given by:

```math
G(f) = G(0) e^{-\pi f^2} 
```

- All we need to do now to find ``G(f)`` is figure out what ``G(0)`` is.
- ``G(0)`` is simply the average value of ``g(t)``, because if you substitute ``f=0`` into the equation for ``G(f)`` the complex exponential term goes away.
    - The integral has actually an [elegant solution](https://en.wikipedia.org/wiki/Gaussian_integral). The result is:
 
```math
G(0) = \int_{-\infty}^\infty e^{-\pi t^2} dt = 1
```

- Therefore:

```math
G(f) = \mathfrak{F} \{  g(t) \} = e^{-\pi f^2} 
```
"""

# ╔═╡ 4a49d01f-0138-4ecd-b5af-a9328cfcbd5c
md"""
### Credits
***

This notebook contains material obtained by [https://towardsdatascience.com/a-proof-of-the-central-limit-theorem-8be40324da83](https://towardsdatascience.com/a-proof-of-the-central-limit-theorem-8be40324da83).
"""

# ╔═╡ 9ed9be7b-e2cc-40a0-a4e1-1298a71a8cc3
cm"""
## Course Flow
***

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
      <td><a href="./open?path=Lectures/Lecture - Spectral Analysis/Lecture-SpectralAnalysis.jl">Spectral analysis</a></td>
    <td><a href="./open?path=Lectures/Lecture - Spectral Analysis/Lecture-SpectralAnalysis.jl">Spectral analysis</a></td>
  </tr>
 </table>


"""

# ╔═╡ 72a28ae9-0fda-4f87-bf1c-e7e133ceec30
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ 7b6c4bfd-786a-40b1-bdd3-eb4570f7d1aa


# ╔═╡ Cell order:
# ╟─d0fdd5a1-5989-471c-86f4-4516645331e6
# ╟─8eac6af6-0e5c-43c7-8781-e97b02d3813f
# ╠═dfdaa2c2-58bc-4911-8e3b-de854e5d56fa
# ╠═b67d9fd5-435e-4150-902e-e6915efdbeaa
# ╠═f419b967-f090-4f5a-9db7-3f9f1a80f8bb
# ╟─d34ab604-27fa-4877-a8e1-d17bb8fb026f
# ╟─48232ae4-81b0-4cf4-80a9-1a0085ba8d9f
# ╟─e5b7c371-b30d-44d3-8ee9-f347719ba366
# ╠═fc298404-78c0-49b9-8250-091dbc2b7487
# ╟─9b5f3662-37da-4e19-a4a3-a80dbbfb041a
# ╟─4a49d01f-0138-4ecd-b5af-a9328cfcbd5c
# ╟─9ed9be7b-e2cc-40a0-a4e1-1298a71a8cc3
# ╟─72a28ae9-0fda-4f87-bf1c-e7e133ceec30
# ╠═7b6c4bfd-786a-40b1-bdd3-eb4570f7d1aa
